/*
  PocketNES-k compilation tool.

  This program is licensed under the GPL.

  April
    v1.0 - initial release
  May
    v1.1 - made to compile with mingw
  June 19, 2004
    v1.2 - change maximal size to 192K+16 since I'm not really
           sure that 192K+17 to 205K will actually work
  */


#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "../minilzo.107/minilzo.h"

// O_BINARY isn't needed in Linux
#ifndef O_BINARY
#define O_BINARY 0
#endif

typedef unsigned long u32;
typedef unsigned short u16;
typedef unsigned char u8;

void usage(char *);
void write_romheader(int out, char *buffer, int size, char *name);
void checksum(int out, u8 *);
void write_compressed_file(int out, char *buffer, int size, int with_size);
void write_u32(int out, u32 num);
void write_u16(int out, u16 num);
void write_u8(int out, u8 num);

const char NESID[] = { 'N', 'E', 'S', '\x1A' };

int main(int argc, char *argv[])
{
    struct stat buf, buf2;
    lzo_uint size;
    int in, out, rom, i, splash_screen, compress;

    if (argc < 2) {
	usage(argv[0]);
	return 1;
    }

    i = 1;
    if ((compress = strcmp(argv[i], "-n")) == 0)
       i++;
    if ((splash_screen = strcmp(argv[i], "-s")) == 0)
       i += 2;

    i += 2;
    if ((strcmp(argv[1], "-mb") == 0) && argc == 5) {
	in = open(argv[3], O_RDONLY | O_BINARY);
	rom = open(argv[4], O_RDONLY | O_BINARY);
	fstat(rom, &buf2);
	fstat(in, &buf);
	if (buf2.st_size + buf.st_size > 255 * 1024) {
	    fprintf(stderr, "%s must be less than %dbytes to multiboot.\n",
		    argv[4], 255 * 1024 - buf.st_size);
	    return 1;
	}
	out = open(argv[2], O_TRUNC | O_WRONLY | O_CREAT | O_BINARY, 0700);
	size = buf.st_size;
	{
	    char buffer[size], *s;
	    int j, n;

	    n = size;
	    s = buffer;
	    while ((j = read(in, s, n)) > 0 && j != n) {
		s += j;
		n -= j;
	    }
	    write(out, buffer, size);
	}
	close(in);
	size = buf2.st_size;
	{
	    char buffer[size], *s;
	    int j, n;

	    n = size;
	    s = buffer;
	    while ((j = read(rom, s, n)) > 0 && j != n) {
		s += j;
		n -= j;
	    }
	    write_romheader(out, buffer, size, argv[i]);
	    write(out, buffer, size);
	}
	close(rom);
	close(out);
	return 0;
    } else if (argc > i) {

	lzo_init();
	
	printf("%s %s\n", argv[i-2], argv[i-1]);
	out = open(argv[i-2], O_TRUNC | O_WRONLY | O_CREAT | O_BINARY, 0700);
        in = open(argv[i-1], O_RDONLY | O_BINARY);
	fstat(in, &buf);

	size = buf.st_size;
	{
	    char buffer[size], *s;
	    int j, n;

	    n = size;
	    s = buffer;
	    while ((j = read(in, s, n)) > 0 && j != n) {
		s += j;
		n -= j;
	    }
	    write(out, buffer, size);
	}
	close(in);

	printf("Size: %d %d\n", size, size & 3);
	while (((size++) & 3) != 0)
	    write_u8(out, 0);

	// Splash screen.
	if (splash_screen == 0) {
	    char buffer[76800];
	    char buffer2[76800 * 102 / 100], workspace[65536];

	    in = open(argv[i-3], O_RDONLY | O_BINARY);
	    fstat(in, &buf);
	    if (buf.st_size != 76800) {
		fprintf(stderr,
			"%s is the wrong size.  Use a valid splash screen.\n",
			argv[2]);
		exit(1);
	    }

	    {
		char *s;
		int j, n;

		n = 76800;
		s = buffer;
		while ((j = read(in, s, n)) > 0 && j != n) {
		    s += j;
		    n -= j;
		}

	    }
	    close(in);

	    lzo1x_1_compress(buffer, buf.st_size, buffer2, &size,
			     workspace);
	    write_u32(out, size);
	    write(out, buffer2, size);
	    while (((size++) & 3) != 0)
		write_u8(out, 0);

	} else {
	    write_u32(out, 0);
	}

	// rom count
	write_u32(out, (u32) (argc - i));

	// add roms
	for (; i < argc; i++) {
	    in = open(argv[i], O_RDONLY | O_BINARY);
	    fstat(in, &buf);
	    {
		char buffer[buf.st_size], *s;
		int j, n;

		n = buf.st_size;
		s = buffer;
		while ((j = read(in, s, n)) > 0 && j != n) {
		    s += j;
		    n -= j;
		}

		close(in);

		// Verify as NES rom through header.
		if (memcmp(buffer, NESID, 4) != 0) {
		    fprintf(stderr, "%s is not an nes rom.\n", argv[i]);
		    exit(1);
		}
		if (!compress || (buf.st_size > 192 * 1024 + 16)) {
		    size = buf.st_size;
		    write_romheader(out, buffer, size, argv[i]);
		    write(out, buffer, size);
		    while (((size++) & 3) != 0)
			write_u8(out, 0);
		} else {
		    char buffer2[buf.st_size * 102 / 100],
			workspace[65536];
		    char *name;
		    int name_length = strlen(argv[i]);

		    // Modify name to end with z.
		    name = malloc(name_length + 1);
		    if (!name) {
			fprintf(stderr, "Error allocating name.\n");
			exit(1);
		    } else if (name_length < 1) {
			fprintf(stderr,
				"Filename must be at least one character.\n");
		    }
		    strcpy(name, argv[i]);
		    if (name[name_length - 1] == 'S') {
			name[name_length - 1] = 'Z';
		    } else {
			name[name_length - 1] = 'z';
		    }
		    lzo1x_1_compress(buffer, buf.st_size, buffer2, &size,
				     workspace);

		    // Notice, pass compressed size, but non-compressed rom.
		    write_romheader(out, buffer, size, name);
		    free(name);

		    write(out, buffer2, size);
		    while (((size++) & 3) != 0)
			write_u8(out, 0);
		}
	    }
	}
	close(out);
	return 0;
    } else {
	usage(argv[0]);
    }

    return 1;
}

void usage(char *prgm)
{
    fprintf(stderr,
	    "Usage: %s [-s <splash_screen.raw>] <out.gba> pocketnes.gba <in1.nes> ...\n",
	    prgm);
}

void write_romheader(int out, char *buffer, int size, char *name)
{
    char title[32], *basename, *extension;
    int i;

    memset(title, 0, 32);
    basename = strrchr(name, '/');
    if (basename == NULL)
	basename = name;
    else
	basename++;
    // title
    i = strlen(basename);
    if (i == 0) {
	fprintf(stderr, "You must specify a file, not a dir: %s\n", name);
	exit(1);
    } else if (i > 31) {
	strncpy(title, basename, 31);
	extension = strrchr(basename, '.');
	if ((i = (i - ((int) extension - (int) basename))) > 31) {
	    fprintf(stderr, "Use a shorter extension for %s\n", name);
	    exit(1);
	}
	strcpy(&title[31 - i], extension);
	title[31] = 0;
    } else {
	strcpy(title, basename);
    }
    write(out, &title, 32);

    //filesize
    write_u32(out, size);

    // flags
    write_u32(out, 0);

    // spritefollow
    write_u32(out, 0);

    // checksum
    checksum(out, buffer);
}

void checksum(int out, u8 * p)
{
    u32 sum = 0;
    int i;

    for (i = 0; i < 128; i++) {
	sum += *p | (*(p + 1) << 8) | (*(p + 2) << 16) | (*(p + 3) << 24);
	p += 128;
    }
    write_u32(out, sum);
}

void write_u32(int out, u32 num)
{
    write(out, &num, 4);
}

void write_u16(int out, u16 num)
{
    write(out, &num, 2);
}

void write_u8(int out, u8 num)
{
    write(out, &num, 1);
}
